import { OtpMode, OtpQualifier } from "./base"

/**
 * Time stand separated by '-' in the format 'Year-Month-Day' when locale is set to 'en'.
 */
export type OtpDate = `${number}-${number}-${number}`

/**
 * Time to depart or arrive
 * @example Format hh:mm:ss
 */
export type OtpTime = `${number}:${number}:${number}`

type InputCoordinatesLegacyLatLng = `${string}::${number},${number}`
type InputCoordinatesLegacyStopId = `${string}::${string}:${number}`

/**
 * he place where the itinerary begins in format name::place, where place is either a lat,lng pair or a stop id. 
 * Use either this argument or from, but not both.
 * 
 * @example Pasila::60.199041,24.932928
 * @example Pasila::HSL:1000202
 */
type InputCoordinatesLegacy =
    | InputCoordinatesLegacyLatLng
    | InputCoordinatesLegacyStopId

interface InputCoordinates {
    address?: string,
    lat: number,
    lon: number,
    locationSlack?: number,
}

interface InputTriangle {
    safetyFactor?: number,
    slopeFactor?: number,
    timeFactor?: number,
}

interface ParkingFilterOperation {
    tags: string[]
}

interface ParkingFilter {
    not: ParkingFilterOperation[]
    select: ParkingFilterOperation[]
}

interface VehicleParkingInput {
    filters?: ParkingFilter[],
    preferred?: ParkingFilter[],
    unpreferredCost?: number,
}

type OptimizeType =
    | "QUICK"
    | "SAFE"
    | "FLAT"
    | "GREENWAYS"
    | "TRIANGLE"

interface InputModeWeight {
    AIRPLANE: number,
    BUS: number,
    CABLE_CAR: number,
    FERRY: number,
    FUNICULAR: number,
    GONDOLA: number,
    RAIL: number,
    SUBWAY: number
    TRAM: number,
}

export interface TransportMode {
    mode: OtpMode,
    qualifier?: OtpQualifier,
}

interface InputUnpreferred {
    agencies?: string,
    routes?: string,
    /**
     * An cost function used to calculate penalty for an unpreferred route/agency. Function should return number of seconds
     * that we are willing to wait for unpreferred route/agency. * String must be of the format: `A + B x`, where A is fixed 
     * penalty and B is a multiplier of transit leg travel time x.
     * 
     * @example `600 + 2.0 x`
     */
    unpreferredCost?: string,
}

/**
 * CURRENT ISSUE WITH BANNED ROUTES AND FLEX ROUTES, OPTION IS IGNORED
 * 
 * WE CURRENTLY HAVE MIDDLEWARE SET UP IN THE generate-planner-query.graphql.ts FILE TO MANUALLY
 * FILTER RESPONSES
 * 
 * RESPONSES ARE EXPECTED TO HAVE AGENCIES/ROUTES/TRIPS AND THEIR `GTFSID` TO BE USED WITH
 * THE BANNED ROUTE MIDDLEWARE
 */
export interface InputBanned {
    /**
     * CURRENT ISSUE WITH BANNED ROUTES AND FLEX ROUTES, OPTION IS IGNORED
     * 
     * WE CURRENTLY HAVE MIDDLEWARE SET UP IN THE generate-planner-query.graphql.ts FILE TO MANUALLY
     * FILTER RESPONSES
     * 
     * RESPONSES ARE EXPECTED TO HAVE AGENCIES AND THEIR `GTFSID` TO BE USED WITH
     * THE BANNED ROUTE MIDDLEWARE
     */
    agencies?: string,
    /**
     * CURRENT ISSUE WITH BANNED ROUTES AND FLEX ROUTES, OPTION IS IGNORED
     * 
     * WE CURRENTLY HAVE MIDDLEWARE SET UP IN THE generate-planner-query.graphql.ts FILE TO MANUALLY
     * FILTER RESPONSES
     * 
     * RESPONSES ARE EXPECTED TO HAVE ROUTES AND THEIR `GTFSID` TO BE USED WITH
     * THE BANNED ROUTE MIDDLEWARE
     */
    routes?: string,
    /**
     * CURRENT ISSUE WITH BANNED ROUTES AND FLEX ROUTES, OPTION IS IGNORED
     * 
     * WE CURRENTLY HAVE MIDDLEWARE SET UP IN THE generate-planner-query.graphql.ts FILE TO MANUALLY
     * FILTER RESPONSES
     * 
     * RESPONSES ARE EXPECTED TO HAVE TRIPS AND THEIR `GTFSID` TO BE USED WITH
     * THE BANNED ROUTE MIDDLEWARE
     */
    trips?: string,
    otherThanPreferredRoutesPenalty?: number,
}

interface InputPreferred {
    agencies?: string,
    routes?: string,
    trips?: string,
}

interface PlanQueryParamsBase {
    date?: OtpDate,
    time?: OtpTime,
    wheelchair?: boolean,
    numItineraries?: number,
    /**
     * CURRENT ISSUE WITH SEARCH WINDOW AND FLEX ROUTES, OPTION IS IGNORED
     * 
     * WE CURRENTLY HAVE MIDDLEWARE SET UP IN THE generate-planner-query.graphql.ts FILE TO MANUALLY
     * FILTER RESPONSES
     * 
     * RESPONSES ARE EXPECTED TO HAVE startTime TO BE USED WITH
     * THE SEARCH WINDOW MIDDLEWARE
     */
    searchWindow?: number,
    pageCursor?: string,
    bikeReluctance?: number,
    bikeWalkingReluctance?: number,
    carReluctance?: number,
    waitReluctance?: number,
    walkSpeed?: number,
    bikeSpeed?: number,
    bikeSwitchTime?: number,
    bikeSwitchCost?: number,
    optimize?: OptimizeType,
    triangle?: InputTriangle,
    arriveBy?: boolean,
    preferred?: InputPreferred,
    unpreferred?: InputUnpreferred,
    walkBoardCost?: number,
    bikeBoardCost?: number,
    banned?: InputBanned,
    transferPenalty?: number,
    transportModes?: TransportMode[] | TransportMode,
    // mode: string,
    modeWeight?: InputModeWeight,
    debugItineraryFilter?: boolean,
    allowKeepingRentedBicycleAtDestination?: boolean,
    keepingRentedBicycleAtDestinationCost?: number,
    boardSlack?: number,
    alignSlack?: number,
    minTransferTime?: number,
    nonpreferredTransferPenalty?: number,
    maxTransfers?: number,
    startTransitStopId?: string,
    omitCanceled?: boolean,
    ignoreRealtimeUpdates?: boolean,
    locale?: "en",
    allowTicketTypes?: string[],
    allowedVehicleRentalNetworks?: string[],
    bannedVehicleRentalNetworks?: string[],
    walkSafetyFactor?: number,
    parking?: VehicleParkingInput,

    /**
     * Domain specific field (not passed to OTP) for filtering with custom middleware (made to mimic searchWindow which acts differently for flex and normal routes)
     * 
     * __*Value is expected as Date of last expected time*__
     * 
     * @example 100 (100 seconds)
     * @example 3600 (1 hour)
     */
    searchWindowMiddleware?: number,

    /**
     * Domain specific field (not passed to OTP) for filtering with custom middleware
     * 
     * __*Max number of allowed routes a user can expect on a single trip.*__
     */
    maxRouteCountMiddleware?: number,
}

/**
 * Plan route arguments based on using "fromPlace" and "toPlace".  If setting
 * "fromPlace", "from" must not be set and vice versa.
 * 
 * @see https://docs.opentripplanner.org/api/dev-2.x/graphql-gtfs/queries/plan
 */
export interface PlanQueryParamsFromPlaceAndToPlace extends PlanQueryParamsBase {
    fromPlace: InputCoordinatesLegacy,
    toPlace: InputCoordinatesLegacy,
}

/**
 * Plan route arguments based on using "from" and "to".  If setting
 * "fromPlace", "from" must not be set and vice versa.
 * 
 * @see https://docs.opentripplanner.org/api/dev-2.x/graphql-gtfs/queries/plan
 */
export interface PlanQueryParamsFromAndTo extends PlanQueryParamsBase {
    from: InputCoordinates,
    to: InputCoordinates,
}

export type PlanQueryParams =
    | PlanQueryParamsFromPlaceAndToPlace
    | PlanQueryParamsFromAndTo